# Kora Payment SDK Examples

This directory contains example implementations for integrating the Kora Payment SDK.

## Examples

### V1 API Examples (Redirect-based)

- **`basic-integration.html`** - Basic V1 redirect-based payment flow
- **`vanilla-js-example.html`** - Vanilla JavaScript example with V1 API
- **`react-example.jsx`** - React component example with V1 API
- **`vue-example.vue`** - Vue.js component example with V1 API

### V2 API Examples (Embedded)

- **`v2-embedded-payment.html`** - Complete V2 embedded payment flow example
  - Shows integration for all three gateways (Checkout.com, Adyen, QiCard)
  - Demonstrates `createEmbeddedSession()`, `initializeCheckoutFlow()`, `initializeAdyenCheckout()`, and `openQiCardPayment()`
  - Gateway is automatically selected based on configuration

## V2 Embedded Payment Flow

The V2 API allows you to embed payment forms directly in your page instead of redirecting to external payment pages.

### Basic Flow

1. **Create Embedded Session**
   ```javascript
   const session = await kora.createEmbeddedSession({
     order_id: 'order_123',
     amount: 10000,
     currency: 'USD',
     payer: { email: 'customer@example.com', name: 'John Doe' },
     billing: { /* ... */ },
     return_url: 'https://merchant.com/success',
     cancel_url: 'https://merchant.com/cancel'
   });
   ```

2. **Initialize Payment Form (Automatic Gateway Detection)**

   **Recommended: Use `initializePayment()` - SDK automatically detects gateway:**
   ```javascript
   await kora.initializePayment({
     session: session,
     container: '#payment-container',
     onSuccess: (result) => {
       console.log('Payment successful!', result);
     },
     onError: (error) => {
       console.error('Payment failed:', error);
     }
     // Note: For Adyen Sessions flow, onSubmit/onAdditionalDetails are NOT needed
     // The Drop-in handles payment automatically using session data
   });
   ```

   **Alternative: Manual Gateway-Specific Initialization**

   If you need more control, you can manually initialize based on gateway:

   **Checkout.com:**
   ```javascript
   await kora.initializeCheckoutFlow({
     session: session,
     container: '#checkout-container',
     onSuccess: (result) => { /* ... */ },
     onError: (error) => { /* ... */ }
   });
   ```

   **Adyen (Sessions Flow):**
   ```javascript
   // Sessions flow: No need to fetch payment methods - Drop-in handles everything
   await kora.initializeAdyenCheckout({
     session: session,
     container: '#adyen-container',
     onSuccess: (result) => {
       console.log('Payment successful!', result);
     },
     onError: (error) => { /* ... */ }
   });
   // Note: Drop-in handles payment automatically using session data
   // No manual onSubmit/onAdditionalDetails/processAdyenPayment needed
   ```

   **QiCard:**
   ```javascript
   kora.openQiCardPayment({
     session: session,
     container: '#qicard-container',
     onSuccess: (result) => { /* ... */ },
     onError: (error) => { /* ... */ }
   });
   ```

## Key Differences: V1 vs V2

### V1 API (Redirect-based)
- Uses `createSession()` → returns `redirect_url`
- User is redirected to external payment page
- Payment completed on gateway's domain
- Uses `redirectToPayment()` and `handleRedirectReturn()`

### V2 API (Embedded)
- Uses `createEmbeddedSession()` → returns session with gateway config
- Payment form embedded directly in your page
- Payment completed on your domain
- Uses gateway-specific initialization methods
- Gateway is selected automatically at session creation

## Authentication

Both V1 and V2 APIs use the same authentication:
- **API Key**: Required in SDK initialization
- **Format**: `pk_test_xxxxx` (public/test) or `pk_live_xxxxx` (public/live)
- **Header**: Automatically handled by SDK (uses `Authorization: Bearer <key>`)

## Running Examples

1. **Build the SDK** (if using from source):
   ```bash
   npm run build
   ```

2. **Update import path** in examples:
   ```javascript
   // Change from:
   import Kora from '../dist/kora-sdk.esm.js';
   
   // To your actual SDK path or CDN URL
   ```

3. **Set your API key** in the example HTML/JS files

4. **Open in browser** (for HTML examples) or run your React/Vue app

## Notes

- Examples use placeholder/demo code - uncomment actual SDK calls when ready
- API keys should be kept secure (use environment variables in production)
- For production, always verify signatures server-side
- Test with sandbox credentials before going live

## Support

For more information, see:
- [SDK Documentation](../docs/)
- [API Reference](../docs/API%20Reference.md)
- [Quick Start Guide](../docs/Quick%20Start%20Guide.md)
