import React, { useState, useEffect } from 'react';
import Kore from 'kora-payment-sdk';

/**
 * React component example for Kore Payment SDK
 */
function PaymentForm() {
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [success, setSuccess] = useState(null);
  const [formData, setFormData] = useState({
    orderId: `order_${Date.now()}`,
    amount: 10000,
    currency: 'USD',
    email: 'customer@example.com',
    name: 'John Doe',
    phone: '',
    address1: '123 Main St',
    address2: '',
    city: 'New York',
    state: 'NY',
    postalCode: '10001',
    country: 'US'
  });

  const [kora] = useState(() => new Kore({
    apiKey: process.env.REACT_APP_KORA_API_KEY || 'pk_test_xxxxxxxxxxxxx',
    environment: process.env.REACT_APP_KORA_ENV || 'sandbox'
  }));

  // Handle return from payment gateway
  useEffect(() => {
    const handlePaymentReturn = async () => {
      const urlParams = new URLSearchParams(window.location.search);
      
      // Check if we have redirect parameters
      if (urlParams.has('status') || urlParams.has('payment_id') || urlParams.has('order_id')) {
        // Note: In production, API secret should be retrieved from server-side
        // For demo purposes, we'll show the structure
        const apiSecret = process.env.REACT_APP_KORA_API_SECRET || 'YOUR_API_SECRET';
        
        try {
          // Handle redirect return with signature verification
          const result = await kora.handleRedirectReturn(urlParams, apiSecret);
          
          if (result.success) {
            // Payment successful
            const { payment_id, order_id, status, gateway } = result.data;
            setSuccess(`Payment successful! Payment ID: ${payment_id}, Order ID: ${order_id}, Status: ${status}${gateway ? `, Gateway: ${gateway}` : ''}`);
          } else {
            // Payment failed
            const { code, message, payment_id, order_id, gateway } = result.error;
            setError(`Payment failed: ${message} (Code: ${code})${payment_id ? `, Payment ID: ${payment_id}` : ''}${order_id ? `, Order ID: ${order_id}` : ''}`);
            
            // Handle specific error codes
            if (code === Kore.ErrorCodes.GATEWAY_SESSION_CREATION_FAILED || 
                code === Kore.ErrorCodes.GATEWAY_TIMEOUT) {
              // Retryable errors - could show retry button
              console.log('Retryable error - user can try again');
            } else if (code === Kore.ErrorCodes.GATEWAY_CREDENTIALS_ERROR || 
                       code === Kore.ErrorCodes.NO_ACTIVE_API_KEY) {
              // Configuration errors - contact support
              console.log('Configuration error - contact support');
            } else if (code === Kore.ErrorCodes.SESSION_NOT_FOUND) {
              // Session expired - redirect to create new session
              console.log('Session expired - create new session');
            }
          }
        } catch (err) {
          setError(`Error processing redirect: ${err.message}`);
        }
      }
    };
    
    handlePaymentReturn();
  }, [kora]);

  const handleChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);
    setError(null);
    setSuccess(null);

    try {
      const session = await kora.createSession({
        order_id: formData.orderId,
        amount: parseInt(formData.amount),
        currency: formData.currency,
        payer: {
          email: formData.email,
          name: formData.name,
          ...(formData.phone && { phone: formData.phone })
        },
        billing: {
          address_line_1: formData.address1,
          ...(formData.address2 && { address_line_2: formData.address2 }),
          city: formData.city,
          ...(formData.state && { state: formData.state }),
          postal_code: formData.postalCode,
          country: formData.country
        },
        return_url: window.location.origin + '/success',
        cancel_url: window.location.origin + '/cancel'
      });

      kora.redirectToPayment(session.redirect_url);
    } catch (err) {
      setError(err.message || 'An error occurred');
      setLoading(false);
    }
  };

  return (
    <div style={styles.container}>
      <h1 style={styles.title}>💳 Kore Payment</h1>
      <p style={styles.subtitle}>Complete your payment securely</p>

      {error && (
        <div style={styles.error}>
          {error}
        </div>
      )}

      {success && (
        <div style={styles.success}>
          {success}
        </div>
      )}

      <form onSubmit={handleSubmit} style={styles.form}>
        <div style={styles.formGroup}>
          <label style={styles.label}>Order ID</label>
          <input
            type="text"
            name="orderId"
            value={formData.orderId}
            onChange={handleChange}
            required
            style={styles.input}
          />
        </div>

        <div style={styles.row}>
          <div style={styles.formGroup}>
            <label style={styles.label}>Amount (cents)</label>
            <input
              type="number"
              name="amount"
              value={formData.amount}
              onChange={handleChange}
              required
              min="1"
              style={styles.input}
            />
          </div>
          <div style={styles.formGroup}>
            <label style={styles.label}>Currency</label>
            <select
              name="currency"
              value={formData.currency}
              onChange={handleChange}
              required
              style={styles.input}
            >
              <option value="USD">USD</option>
              <option value="EUR">EUR</option>
              <option value="GBP">GBP</option>
            </select>
          </div>
        </div>

        <h3 style={styles.sectionTitle}>Payer Information</h3>

        <div style={styles.formGroup}>
          <label style={styles.label}>Email</label>
          <input
            type="email"
            name="email"
            value={formData.email}
            onChange={handleChange}
            required
            style={styles.input}
          />
        </div>

        <div style={styles.formGroup}>
          <label style={styles.label}>Name</label>
          <input
            type="text"
            name="name"
            value={formData.name}
            onChange={handleChange}
            required
            style={styles.input}
          />
        </div>

        <div style={styles.formGroup}>
          <label style={styles.label}>Phone (Optional)</label>
          <input
            type="tel"
            name="phone"
            value={formData.phone}
            onChange={handleChange}
            style={styles.input}
          />
        </div>

        <h3 style={styles.sectionTitle}>Billing Address</h3>

        <div style={styles.formGroup}>
          <label style={styles.label}>Address Line 1</label>
          <input
            type="text"
            name="address1"
            value={formData.address1}
            onChange={handleChange}
            required
            style={styles.input}
          />
        </div>

        <div style={styles.formGroup}>
          <label style={styles.label}>Address Line 2 (Optional)</label>
          <input
            type="text"
            name="address2"
            value={formData.address2}
            onChange={handleChange}
            style={styles.input}
          />
        </div>

        <div style={styles.row}>
          <div style={styles.formGroup}>
            <label style={styles.label}>City</label>
            <input
              type="text"
              name="city"
              value={formData.city}
              onChange={handleChange}
              required
              style={styles.input}
            />
          </div>
          <div style={styles.formGroup}>
            <label style={styles.label}>State</label>
            <input
              type="text"
              name="state"
              value={formData.state}
              onChange={handleChange}
              style={styles.input}
            />
          </div>
        </div>

        <div style={styles.row}>
          <div style={styles.formGroup}>
            <label style={styles.label}>Postal Code</label>
            <input
              type="text"
              name="postalCode"
              value={formData.postalCode}
              onChange={handleChange}
              required
              style={styles.input}
            />
          </div>
          <div style={styles.formGroup}>
            <label style={styles.label}>Country</label>
            <select
              name="country"
              value={formData.country}
              onChange={handleChange}
              required
              style={styles.input}
            >
              <option value="US">US</option>
              <option value="GB">GB</option>
              <option value="CA">CA</option>
            </select>
          </div>
        </div>

        <button
          type="submit"
          disabled={loading}
          style={styles.button}
        >
          {loading ? 'Processing...' : 'Pay Now'}
        </button>
      </form>
    </div>
  );
}

const styles = {
  container: {
    maxWidth: '500px',
    margin: '0 auto',
    padding: '40px',
    background: 'white',
    borderRadius: '12px',
    boxShadow: '0 20px 60px rgba(0, 0, 0, 0.3)'
  },
  title: {
    fontSize: '28px',
    color: '#333',
    marginBottom: '10px'
  },
  subtitle: {
    color: '#666',
    marginBottom: '30px',
    fontSize: '14px'
  },
  form: {
    display: 'flex',
    flexDirection: 'column'
  },
  formGroup: {
    marginBottom: '20px'
  },
  row: {
    display: 'grid',
    gridTemplateColumns: '1fr 1fr',
    gap: '15px'
  },
  label: {
    display: 'block',
    marginBottom: '8px',
    color: '#333',
    fontWeight: '500',
    fontSize: '14px'
  },
  input: {
    width: '100%',
    padding: '12px',
    border: '2px solid #e0e0e0',
    borderRadius: '8px',
    fontSize: '14px'
  },
  sectionTitle: {
    margin: '30px 0 20px',
    color: '#333',
    fontSize: '18px'
  },
  button: {
    width: '100%',
    padding: '14px',
    background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
    color: 'white',
    border: 'none',
    borderRadius: '8px',
    fontSize: '16px',
    fontWeight: '600',
    cursor: 'pointer',
    marginTop: '10px'
  },
  error: {
    background: '#fee',
    color: '#c33',
    padding: '12px',
    borderRadius: '8px',
    marginBottom: '20px',
    fontSize: '14px'
  },
  success: {
    background: '#efe',
    color: '#3c3',
    padding: '12px',
    borderRadius: '8px',
    marginBottom: '20px',
    fontSize: '14px'
  }
};

export default PaymentForm;

